﻿using System.Drawing;
using System.Security.Policy;
using System.Windows.Forms;
using Edu.Library;
using Edu.Library.Logo;
using Edu.WinForms.EsyWin.Forms;

namespace Edu.WinForms.EsyWin
{
    public class EduInOut : IEduInOut
    {
        const string NEWLINE = "\r\n";

        private PictureBox _screen;
        private RichTextBox _output;
   

        public EduInOut(PictureBox screen, RichTextBox output)
        {
            _screen = screen;
            _output = output;
        }

        private string AddLine(string text)
        {
            return text + NEWLINE;
        }

        public void ClearTextArea()
        {
            _output.Clear();
            _output.Refresh();
        }

        public void Clear()
        {
            _output.Clear();
            ClearScreen();
        }

        public void Write(string text)
        {
            var outputText = AddLine(text);
            _output.AppendText(outputText);
            _output.SelectionStart = _output.Text.Length;
            _output.ScrollToCaret();
            _output.Refresh();
            _output.Focus();
        }


        public void MessageBox(string message)
        {
            using (var dialog = new MessageDialog())
            {
                dialog.ShowMessage(message);
            }
        }

        public bool ConfirmBox(string message)
        {
            using (var dialog = new MessageDialog())
            {
                var confirm = dialog.ShowConfirm(message);
                return confirm;
            }
        }

        public string ReadText(string message)
        {
            using (var inputDialog = new InputTextDialog())
            {
                var text = inputDialog.ReadText(message);
                return text;
            }
        }

        public int ReadInteger(string message)
        {
            using (var inputDialog = new InputIntDialog())
            {
                var no = inputDialog.ReadInteger(message);
                return no;
            }
        }

        public bool ReadBoolean(string message)
        {
            using (var inputDialog = new InputBooleanDialog())
            {
                var answer = inputDialog.ReadBoolean(message);
                return answer;
            }
        }

        public static PointF ToCenterScreen(float w, float h, PointF point)
        {
            var p = new PointF(point.X + w / 2, h/2 - point.Y);
            return p;
        }

        public void ClearScreen(Color? color = null)
        {
            if (color == null) color = Color.White;
            using (var g = _screen.CreateGraphics())
            {
                g.Clear(color.Value);
            }
        }

        public void Draw(Line line)
        {
            if (line == null) return;

            using (var g = _screen.CreateGraphics())
            {
                Draw(g, line);
            }
        }

        public void Draw(Graphics g, Line line)
        {
            if (line == null) return;

            var w = _screen.Width;
            var h = _screen.Height;

            using (var pen = new Pen(line.PenColor, 2))
            {
                var pa = ToCenterScreen(w, h, line.Pa);
                var pb = ToCenterScreen(w, h, line.Pb);
                g.DrawLine(pen, pa, pb);
            }
        }

        public void Draw(List<Line> lines, Color? bgColor = null, bool clearScreen = true)
        {
            if (lines == null) lines = new List<Line>();
            if (bgColor == null) bgColor = Color.White;

            using (var g = _screen.CreateGraphics())
            {
                if (clearScreen)
                {
                    g.Clear(bgColor.Value);
                }

                foreach(var line in lines)
                {
                    Draw(g, line);
                }
            }
        }
    }
}